''' QAPDFs: Combine and prepare proton Pt QA reports
Author: duvall@wustl.edu
'''


# SETTINGS
# ENABLE_GHOSTSCRIPT = True   #KEEPME
ENABLE_GHOSTSCRIPT = False  #TEMP


## INIT

# imports
# built-ins:
import sys, re, io, os, os.path, shutil, stat
import tempfile as tmp
import locale as lc
from importlib import import_module
from pprint import pp
import logging as log
# settings
# log.basicConfig(level = log.WARN)
# log.basicConfig(level = log.INFO)
# log.basicConfig(level = log.DEBUG)
# externals:
if ENABLE_GHOSTSCRIPT:
    ext_libs = [ ('pypdf', 'pdf'), ('ghostscript', 'gs') ] #FIXME: TEMP_DISABLE
else:
    ext_libs = [ ('pypdf', 'pdf') ]
qapdfs_loc = os.path.dirname(__file__)
install_path = '"' + os.path.join(qapdfs_loc, 'install.py') + '"'
import_success = True
for (lib_name, lib_abbr) in ext_libs:
    try:
        lib = import_module(lib_name)
    except ModuleNotFoundError:
        print(f'Module `{lib_name}` not found; you will need to install it before using QAPDFs.')
        import_success = False
    except:
        print('Import Error:', sys.exc_info())
        import_success = False
    else:
        globals()[lib_abbr] = lib
if not import_success:
    print(f'\nError: One or more required modules either not found or failed to load. Try the following:')
    print(f'  1) Exit Python.')
    print(f'  2) Run this command at your shell prompt: `python {install_path}`.')
    print(f'  3) Re-launch Python and try importing QAPDFs again.\n')
    cancel_err_msg = 'Canceling QAPDFs import; see instructions above.'
    raise RuntimeError(cancel_err_msg)
IS_UNIX = (os.name=='posix')


## CLASSES

class Report:
    ''' The primary class for this module.\n
    Attributes
    ----------
    patient_absolute_dir :  path-like
    patient :               .Patient
    field_pdfs :            list of path-like
    home_dir :              path-like
    home_writeable :        bool
    tempdir :               directory on local machine
    tempfile :              intermediate PDF
    spreadsheet_pdf :       path-like
    final_outfile :         path-like
    info :                  str
    Methods
    -------
    import_dir_info() :     parse the `patient_dir` and set parameters accordingly
    show() :                print current parameters
    merge() :               merge documents
    compress() :            compress final result
    run() :                 execute PDF operations (i.e., merge() and compress()) with current parameters
    '''

    # ctor
    def __init__(self):
        self.patient_absolute_dir = os.getcwd()
        self.patient = None
        self.field_pdfs = []
        self.home_dir = None
        self.home_writeable = False
        self.tempdir = None
        self.tempfile = None
        self.spreadsheet_pdf = ''
        self.final_outfile = ''
        self.info = ''
        # attempt to create a local (i.e., non-network) temporary directory
        #   in the user's home directory
        if IS_UNIX:
            self.home_dir = os.getenv('HOME')
        else:
            self.home_dir = os.path.expanduser( '~' + os.getlogin() )
        self.home_writeable = ( os.access(self.home_dir, os.W_OK) )
        if self.home_writeable:
            self.tempdir = tmp.TemporaryDirectory(dir = self.home_dir)
        # fill and show other information:
        self.import_dir_info()
        self.show()

    # import_dir_info
    def import_dir_info(self):
        self.patient = Patient()
        self.spreadsheet_pdf = self.patient.spreadsheet_name
        try:
            os.path.isdir('PDFs')
        except:
            raise FileNotFoundError('Directory `PDFs` not found')
        os.chdir('PDFs')
        pdflist = ls(r'.*\.pdf')
        self.field_pdfs = [ os.path.join('PDFs', _) for _ in pdflist ]
        os.chdir('..')
        log.warn(f'  pwd = {os.getcwd()}\n  spreadsheet_pdf = {self.spreadsheet_pdf}') #debug
        self.final_outfile = self.spreadsheet_pdf.replace('.pdf', '_Combined.pdf')
        self.info += f'Field_PDFs:\n'
        for _ in self.field_pdfs:
            self.info += f'  {_}\n'
        self.info += f'Spreadsheet_PDF:\t{self.spreadsheet_pdf}\n'
        self.info += f'Final_Output_File:\t{self.final_outfile}'

    # show
    def show(self):
        curpar = '\nCURRENT SETTINGS'
        print(f'{curpar}', '='*len(curpar), '', sep='\n')
        self.patient.show()
        print('\nREPORT INFO:')
        print(self.info)

    # merge
    def merge(self):
        zan = 'zAnalysis.pdf'
        zanr = 'zAnalysis_R.pdf'
        print('Merging reports...')
        if self.home_writeable:
            self.tempfile = tmp.NamedTemporaryFile(dir = self.tempdir.name, suffix = '.pdf',  delete = False)
        else:
            self.tempfile = tmp.NamedTemporaryFile(dir = os.getcwd(), suffix = '.pdf',  delete = False)
        final_outpath = os.path.join(self.patient_absolute_dir, self.final_outfile)
        merger = pdf.PdfWriter()
        merger.append(self.spreadsheet_pdf)
        for file in self.field_pdfs:
            merger.append(file)
        merger.write(self.tempfile.name)
        merger.close()
        os.chmod(self.tempfile.name, ALL_CAN_READ | stat.S_IWUSR)
        print('Merging complete.\n')

    # compress
    def compress(self):
        print('Compressing result...')
        encoding = lc.getpreferredencoding()
        # gsargs = f'-sDEVICE=pdfwrite -dCompatibilityLevel=1.4 -dPDFSETTINGS=/prepress -dNOPAUSE -dQUIET -dBATCH -sOutputFile={self.final_outfile} {os.path.basename(self.tempfile.name)}'
        # outfile = os.path.join( self.tempdir.name, self.final_outfile )
        # log.debug(f'  outfile = {outfile}')
        if os.path.isdir(self.tempdir.name):
            os.chdir(self.tempdir.name)
        gsargs = [ 'compress',
            '-sDEVICE=pdfwrite',
            '-dCompatibilityLevel=1.4',
            '-dPDFSETTINGS=/prepress',
            '-dNOPAUSE',
            '-dQUIET',
            '-dBATCH',
            '-dSAFER',
            '-sOutputFile=' + self.final_outfile,
            f'{os.path.basename(self.tempfile.name)}' ]
            # '-sOutputFile=' + outfile,
        enc = lc.getpreferredencoding()
        gsargs_enc = [ a.encode(enc) for a in gsargs ]
        if ENABLE_GHOSTSCRIPT:
            gs.Ghostscript(*gsargs_enc)
        else:
            from pathlib import Path
            Path(self.final_outfile).touch()
        os.chmod(self.final_outfile, ALL_CAN_READ | stat.S_IWUSR)
        os.chdir(self.patient_absolute_dir)
        print('Compression complete.\n')

    # run
    def run(self):
        self.merge()
        self.compress()
        os.remove(self.tempfile.name)
        if os.path.isdir(self.tempdir.name):
            shutil.copy( os.path.join(self.tempdir.name, self.final_outfile), self.patient_absolute_dir )
            self.tempdir.cleanup()
        print(f'Done! The report file "{self.final_outfile}" should now be ready for upload into patient\'s Documents.\n')


class Patient:
    ''' Holds patient info.\n
    Attributes
    ----------
    patient_relative_dir :  path-like
    first :                 str
    last :                  str
    mrn :                   str
    plan_name :             str
    spreadsheet_name :      str
    info :                  str
    Methods
    -------
    fill_from_dir() :       parse the `patient_dir` and set parameters accordingly
    show() :                print current parameters
    '''

    # ctor
    def __init__(self):
        self.patient_relative_dir = ''
        self.first = ''
        self.last = ''
        self.mrn = ''
        self.plan_name = ''
        self.spreadsheet_name = ''
        self.info = ''
        self.fill_from_dir()

    # fill_from_dir
    def fill_from_dir(self):
        cwd = os.getcwd()
        self.plan_name = os.path.basename(cwd)
        patdir = os.path.basename( os.path.dirname(cwd) )
        self.patient_relative_dir = patdir
        split1 = patdir.split(',')
        split2 = split1[1].split('~')
        self.last = split1[0].strip()
        self.first = split2[0].strip()
        self.mrn = split2[1].strip()
        self.spreadsheet_name = self.last.title() + self.first[0] + '_' + self.mrn + '.pdf'
        self.info += f'FirstName\t{self.first}\n'
        self.info += f'LastName\t{self.last}\n'
        self.info += f'MRN\t\t{self.mrn}\n'
        self.info += f'Plan\t\t{self.plan_name}'

    # show
    def show(self):
        print('PATIENT INFO:')
        print(self.info)

    # override __repr__
    def __repr__(self):
        return f'{NAME}.Patient object'

    # override __str__
    def __str_(self):
        return f'{NAME}.Patient object'


## FUNCTIONS

# listre -- regex-search a list
def listre( listname: list, restring: str ) -> list:
    """ Search a list for items matching a regex. """
    pat = re.compile(restring)
    matches = list( filter(pat.match, listname) )
    return matches
# filter-enabled ls()
def ls( regex: str = '', path: str = None ) -> list:
    """ List files matching a given regex in a given path. """
    return listre( os.listdir(path), regex )


## DATA

ALL_CAN_READ = stat.S_IRUSR | stat.S_IRGRP | stat.S_IROTH
NAME = 'QAPDFs'
VERSION = '0.8.0'
# print(f'{NAME} v{VERSION} loaded.\n')

