''' Package for managing proton patient QA.

Given a patient's RTPLAN DICOM file, performs or assists with the following steps:
    - Create pt's (sub)directories on the Q drive
    - Call MATLAB to create pt's spreadsheet
    - Initiate 2nd-dose calculation
    - Merge and compress pt's final-report PDF

Authorship:
duvall@wustl.edu
Siteman Cancer Center,
Washington University School of Medicine
11/2024

'''

## TODO
# GENERAL
# #   - FIXME -- CURRENT: Temporarily disable all GhostScript imports & calls in order to test on NT platforms
# - use DICOM to check fields, field_pdfs, etc.
# - verify shared drives by network address
# - use pdfgrep [or similar] to automate/streamline proofreading of reports
#     (espec. pt_basics and_pass rates)
# MATLAB
# - check on args for calling XLS-creator
# ARIA ?
# - investigate scripting
# MOSTLY DONE
# - implement checks, warnings, etc. for missing or already-existing files / dirs / vars / values
# - add status msgs and instructions for user
# - add GUI
# - deal with network paths
# DONE
# - TOP: implememnt devtools, espec. for preparing production!
# - DONE: implement sub-modules (e.g., qapdfs)
# - add some actual test PDFs


## SETTINGS
# TESTMODE = True             #TEMP
TESTMODE = False          #KEEPME
# GUI = True                #TEMP
GUI = False                 #KEEPME
# CHECK_DRIVES = False      #TEMP
CHECK_DRIVES = True         #KEEPME


## IMPORTS
# builtins
import re, io, sys, os, os.path, glob, shutil, enum, socket
import logging as log
import pathlib #TESTING
if GUI:       #KEEPME?
    from tkinter import simpledialog, filedialog, messagebox       #KEEPME?
# from tkinter import simpledialog, filedialog, messagebox         #TEMP?
from importlib import import_module
from datetime import datetime
from enum import Enum, auto
# from ipaddress import IPv4Address
from pprint import pp
# from this package
from qamgr.qapdfs import Report
# externals
ext_libs = [ ('pydicom', 'pydicom'), ('tabulate', 'tabulate') ]
qamgr_pkg_dir = os.path.dirname(__file__)
install_path = '"' + os.path.join(qamgr_pkg_dir, 'install.py') + '"'
import_success = True
print()
for (lib_name, lib_abbr) in ext_libs:
    try:
        lib = import_module(lib_name)
    except ModuleNotFoundError:
        print(f'Module `{lib_name}` not found; you will need to install it before using QAMgr.')
        import_success = False
    except:
        print('Import Error:', sys.exc_info())
        import_success = False
    else:
        globals()[lib_abbr] = lib
if not import_success:
    print(f'\nError: One or more required modules either not found or failed to load. Try the following:')
    print(f'  1) Exit Python.')
    print(f'  2) Run this command at your shell prompt: `python {install_path}`.')
    print(f'  3) Re-launch Python and try importing QAMgr again.\n')
    cancel_err_msg = 'Canceling QAMgr import; see instructions above.'
    raise RuntimeError(cancel_err_msg)
# settings
log.basicConfig(level = log.WARN)
# log.basicConfig(level = log.INFO)
# log.basicConfig(level = log.DEBUG)
# aliases
HAS_TABULATE = False
if 'tabulate' in os.sys.modules:
    HAS_TABULATE = True
    tab = tabulate.tabulate
else:
    def tab(items: list = [], **kwargs):
        ''' Stand-in for tabulate.tabulate if pkg not available. '''
        return pp(items)
# # windows-only #TEMP_DISABLE
# if os.name == 'nt':
#     try:
#         from pyreadline3 import Readline
#     except ModuleNotFoundError:
#         print(f'Module `pyreadline3` not found; recommend running `pip install pyreadline3` at your shell to make life easier (this should enable tab-completion and other features at the interactive Python prompt).')
#     else:
#         readline = Readline()


## CLASSES

# Steps -- enums for specifying the various parts of QAMgr
class Steps(enum.Enum):
    ''' Specifies the various parts of QAMgr\n
    '''
    # values
    ALL = 0
    DRIVES = enum.auto()
    START = enum.auto()
    CREATE_SUBDIRS = enum.auto()
    CREATE_SPREADSHEET = enum.auto()
    START_2DC = enum.auto()
    PREPARE_REPORT = enum.auto()
    COMPLETE = enum.auto()

S = Steps


# QAMgr -- primary class
class QAMgr:
    ''' Primary class for this module.\n
    Attributes
    ----------
    self.cpa :              str
    os_info :               OSInfo object
    qa_dir :                path-like
    patient :               Patient object
    main_dir :              path-like
    started_ts :            datetime.datetime object
    completed_ts :          datetime.datetime object
    tz :                    datetime.timezone object
    status_mon :            QAStatusMonitor object
    final_basename :        path-like
    dosecalc_dir :          path-like
    topas_pt_dir :          path-like
    Methods
    -------
    create_subdirs() :      create all needed (sub)directories and move input DICOM file
    create_spreadsheet() :  call MATLAB spreadsheet creator
    start_2dc() :			initiate 2nd-dose calculation
    status() :              print current status
    info() :                print summary
    run() :                 perform all actions
    finished() :            announce that QAManager is finished
    rtplan_summary():       shortcut for `self.patient.rtplan.summarize`
    '''

    # ctor
    def __init__(self, FILE = ''):
        # init
        self.cpa = ''
        self.os_info = OSInfo()
        now = datetime.now().replace(microsecond=0).astimezone()
        self.tz = now.tzinfo
        if TESTMODE:
            self.qa_dir = QA_DIR
        else:
            self.qa_dir = self.os_info.Q_base
        self.ask_cpa()
        self.patient = Patient(FILE)
        self.main_dir = os.path.join( self.qa_dir, 'PatientQA', self.patient.patient_dir, self.patient.plan_dir )
        self.started_ts = now.astimezone(self.tz)
        self.completed_ts = None
        self.status_mon = QAStatusMonitor(self)
        self.status_mon.steps[S.START.name] = True
        self.final_basename = os.path.join( self.main_dir, self.patient.final_basename )
        self.dosecalc_dir = ''
        self.topas_pt_dir = ''
        print(f'\nQAMgr created for `{self.main_dir+os.sep}` by {self.cpa} at {self.started_ts.isoformat()}.\n')
        self.info()
        self.status()

    # ask_cpa
    def ask_cpa(self):
        ''' Prompt user (presumably a CPA) to enter their initials. '''
        cpa_prompt = '\nPlease enter your initials below.\nYou will then be prompted to select an input RTPLAN DICOM file.\n'
        if GUI:
            from tkinter import simpledialog
            # self.cpa = simpledialog.askstring('CPA Initials', cpa_prompt).strip()
            initials = simpledialog.askstring('CPA Initials', cpa_prompt)
            # if initials:
            #     self.cpa = initials.strip()
            # else:
            #     msg_no_initials = 'No initials given; QAManager canceled.'
            #     raise ValueError(msg_no_initials)
        else:
            # self.cpa = input(cpa_prompt+'\nCPA Initials:\t').strip().upper()
            initials = input(cpa_prompt+'\nCPA Initials:\t')
        # if not self.cpa:
        if not initials:
            msg_no_cpa = 'Cannot proceed without CPA identification. Exiting...\n'
            # if GUI: #TODO -- will probably need restructuring using if hasattr(self, 'root') as used in FlashLogs
            #     messagebox.showerror(msg_no_cpa)
            raise ValueError(msg_no_cpa)
        else:
            self.cpa = initials.strip().upper()

    # create_subdirs
    def create_subdirs(self):
        ''' Create patient\'s `import`, `plan`, and `PDFs` subdirectories. '''
        print('Creating subdirectories...')
        if TESTMODE:
            path = os.path.join( self.qa_dir, 'PatientQA', self.patient.patient_dir )
        else:
            path = os.path.join( self.qa_dir, self.patient.patient_dir )
        # print(f'First path:\t{path}') #debug
        os.makedirs(path, exist_ok=True)
        path = os.path.join( path, self.patient.plan_dir )
        # print(f'Second path:\t{path}') #debug
        os.makedirs(path, exist_ok=False)
        os.chdir(path)
        [ os.makedirs(subdir) for subdir in ['imports', 'plan', 'PDFs'] ]
        if TESTMODE:
            srcdir = os.path.dirname(self.patient.rtplan.realpath)
            shutil.copy( self.patient.rtplan.realpath, 'plan' ) #SAFE
            patient_final_basename = self.patient.final_basename + '.pdf'
            field_pdfs = ls(r'.*\.pdf', srcdir )
            if 'spreadsheet.pdf' in field_pdfs:
                field_pdfs.remove('spreadsheet.pdf')
            if patient_final_basename in field_pdfs:
                field_pdfs.remove(patient_final_basename)
            log.debug(field_pdfs)
            for _ in field_pdfs:
                shutil.copy(os.path.join(srcdir, _), 'PDFs')  #SAFE
            # copy PATIENT_pdf if needed
            PATIENT_PDF = 'DoeJ_0123456789.pdf'
            # PATIENT_PDF = os.path.join(self.qa_dir, 'DoeJ_0123456789.pdf')
            # log.warn(f'PATIENT_PDF = {PATIENT_PDF}')
            # log.warn(f'  isfile(PATIENT_PDF):\t{os.path.isfile(PATIENT_PDF)}')
            # shutil.copyfile( PATIENT_PDF, os.path.join(self.main_dir, PATIENT_pdf) )
            shutil.copyfile( os.path.join(self.qa_dir, PATIENT_PDF), os.path.join(self.main_dir, PATIENT_PDF) )
        else:
            shutil.move( self.patient.rtplan.realpath, 'plan' ) #SAFE
        self.status_mon.steps[S.CREATE_SUBDIRS.name] = True
        print('Done.')

    # create_spreadsheet
    def create_spreadsheet(self):
        ''' Call MATLAB to create patient\'s spreadsheet. '''
        print('Creating spreadsheet [PLACEHOLDER]...')
        xls_filename = self.final_basename + '.xls'
        if TESTMODE:
            # pathlib.Path(xls_filename).touch()
            # srcdir = os.path.dirname(self.patient.rtplan.realpath)
            # srcdir = os.path.realpath( os.path.join( self.main_dir, '..', '..', 'test_pdfs' ) )
            if IS_UNIX:
                srcdir = '/home/mark/mark_code/qamgr/test/tps'
            else:
                srcdir = r'C:\Users\duvall\Documents\duvall_personal\tests_and_demos\qamgr'
            log.debug(f'   spr_scrdir = {srcdir}')
            testxls = os.path.join(srcdir, os.path.basename(self.final_basename + '.xls') )
            if os.path.isfile(testxls):
                # shutil.copy( testxls, os.path.join(os.getcwd(), '..') ) #NOTSAFE?
                shutil.copy( testxls, os.getcwd() ) #SAFE
            else:
                msg_xls_not_found = f'File `{testxls}` not found.'
                raise FileNotFoundError(msg_xls_not_found)
        self.status_mon.steps[S.CREATE_SPREADSHEET.name] = True
        print('Done.')

    # start 2nd-dose calc
    def start_2dc(self, exdir: str = '', rn_file: str = ''):
        ''' Perform needed file and directory operations for MATLAB
              to automatically initiate patient\'s 2nd dose calculation. '''
        print('Initiating 2nd dose calculation...')
        p = self.patient
        if not exdir:
            exdir = p.mrn
        if TESTMODE:
            self.dosecalc_dir = os.path.join( QA_DIR, '2nd Dose Calc', f'{p.family}, {p.given} {p.plan_name}' )
            # topas_dosecalc_dir = 
        else:
            self.dosecalc_dir = os.path.join( self.os_info.Q.current_path, Q_DIR_PARTS[0], '2nd Dose Calc', f'{p.family}, {p.given} {p.plan_name}' )
        os.makedirs(self.dosecalc_dir)
        if TESTMODE:
            xpath = os.path.join( self.qa_dir, 'EXPORT', exdir )
            if not os.path.isdir(xpath):
                os.makedirs(xpath)
            pathlib.Path(os.path.join(xpath, 'rn_test.dcm')).touch()
        else:
            xpath = os.path.join( self.os_info.X_base, exdir ) #FIXME
            if not os.path.isdir(xpath):
                msg_xpathmissing = f'\nWarning: Folder `{p.mrn}` not found in EXPORTS; please either provide an alternate folder name or export from External Beam Planning and try again.'
                raise UserWarning(msg_xpathmissing)
        rn_files = sorted( ls(r'rn.*\.dcm', path = xpath) )
        L = len(rn_files)
        match L:
            case 0:
                msg_rn_notfound = '\nNo RN files (RN*.dcm) found.'
                raise FileNotFoundError(msg_rn_notfound)
            case 1:
                rn_file = rn_files[0]
            case _:
                msg_rn_multiple = '\nMultiple RTPLAN files (RP*.dcm) files found; please specify one.'
                # raise OSError(msg_rn_multiple)
                if GUI:
                    from tkinter import filedialog #TODO: address 'Cancel' case
                    rn_file = filedialog.askopenfilename()
                else:
                    rn_file = xpath + input('  ')
        if rn_file == []: #FIXME: not caught!
            return
        # rn_realpath = os.path.realpath(rn_file) #KEEPME?
        rn_realpath = os.path.join(xpath, 'rn_test.dcm') #TEMP?
        copied_rn_filepath = os.path.join( self.dosecalc_dir, os.path.basename(rn_file) ) 
        shutil.copy( rn_realpath, self.dosecalc_dir ) #SAFE
        if os.path.isfile(copied_rn_filepath):
            # shutil.rmtree(xpath) #KEEPME #NOTSAFE?
            pass #TEMP
        else:
            msg_rn_notcopied = f'\n`{copied_rn_filepath}` not found; copying from `{xpath}` seems to have failed.'
            raise OSError(msg_rn_notcopied)
        if GUI:
            from tkinter import messagebox
            messagebox.showinfo(title = '2nd Dose Calc',  message = '\nNow waiting for RD DICOM files from RayStation `2nd Dose Calc` script; please click <OK> when they are ready.\n\n')
        else:
            input('\nNow waiting for RD DICOM files from RayStation `2nd Dose Calc` script; please press <Enter> or <Return> when they are ready.\n\n')
        if TESTMODE:
            pathlib.Path(os.path.join(self.dosecalc_dir, 'rd_test.dcm')).touch()
        if len(ls(r'rd.*\.dcm', path = self.dosecalc_dir)) == 0:
            msg_warn_nordfile = f'\nWarning: No RD DICOM files found in `{self.dosecalc_dir}` (there is typically one for each field in the plan); 2nd dose calculation may not succeed.'
            log.warn(msg_warn_nordfile)
        if TESTMODE:
            topas_dir = os.path.join( QA_DIR, *Z_DIR_PARTS )
            if not os.path.isdir(topas_dir):
                os.makedirs(topas_dir)
        else:
            topas_dir = self.os_info.Z_base
        shutil.copy( rn_realpath, copied_rn_filepath ) #NOTSAFE?
        self.topas_pt_dir = topas_dir + os.sep + os.path.basename(self.dosecalc_dir) 
        shutil.copytree( self.dosecalc_dir, self.topas_pt_dir ) #SAFE
        # topas_names = { 'exdir': exdir,                                         #debug
        #         'self.dosecalc_dir': self.dosecalc_dir,
        #         'xpath': xpath,
        #         'rn_files': rn_files,
        #         'rn_file': rn_file,
        #         'rn_realpath': rn_realpath,
        #         'copied_rn_filepath': copied_rn_filepath,
        #         'self.topas_pt_dir': self.topas_pt_dir
        # }
        # print(tab([(k,v) for k,v in topas_names.items()], tablefmt='plain'))    #debug
        if not os.listdir( self.topas_pt_dir ):
            msg_topasfilesmissing = f'\nNo files found in `{self.topas_pt_dir}`; 2nd dose calc will not start.'
            raise FileNotFoundError(msg_topasfilesmissing)
        else:
            self.status_mon.steps[S.START_2DC.name] = True
            print('Done.')

    # prepare final-report PDF
    def prepare_report(self):
        ''' Combine patient\'s spreadsheet and `PDFs` files into a single,
              compressed, final-report PDF ready for uploading into Aria. '''
        print('Preparing final report...')
        # BASIC TEST VERSION:
        # xls_pdf_filename = self.final_basename + '.pdf'
        # if not ( os.path.isfile(xls_pdf_filename) or os.path.islink(xls_pdf_filename) ):
        #         messagebox.showinfo(title = 'Prepare Final Report', message = 'Please use the Excel `File > Export > PDF` feature to create the PDF of the patient\'s spreadsheet.')
        #         if TESTMODE:
        #             pathlib.Path(xls_pdf_filename).touch()
        # pdf_filename = self.final_basename + '_Combined.pdf'
        # if TESTMODE:
        #     pathlib.Path(pdf_filename).touch()
        # self.status_mon.steps[S.PREPARE_REPORT.name] = True
        # # log.info('  Done.')
        # print('Done.')
        # QAPDFS VERSION:
        origwd = os.getcwd()
        if os.path.isdir(self.main_dir):
            os.chdir(self.main_dir)
        else:
            msg_notadir = f'Directory {self.main_dir} does not exist; please run create_subdirs first.'
            raise NotADirectoryError(msg_notadir)
        log.warn(f'  qamgr.py_l373: IS_UNIX = {IS_UNIX}')
        if IS_UNIX:
            if TESTMODE:
                xls_pdf_filename = self.final_basename + '.pdf'
                pdf_filename = self.final_basename + '_Combined.pdf'
                srcdir = os.path.dirname(self.patient.rtplan.realpath)
                testspr_dir = os.path.realpath( os.path.join(srcdir, '..', 'tps') )
                testspr = os.path.join( testspr_dir, os.path.basename(xls_pdf_filename) )
                if os.path.isfile(testspr):
                    shutil.copy( testspr, xls_pdf_filename ) #NOTSAFE?
                # else: #TEMP_DISABLE
                #     log.error(f'  File testspr = `{testspr}` not found.')
            r = Report()
            r.run()
        else:
            pass #debug #FIXME
        self.status_mon.steps[S.PREPARE_REPORT.name] = True
        os.chdir(origwd)
        if not __name__ == '__main__':
            self.status()

    # status
    def status(self, step = S.ALL):
        ''' Shortcut for this object\'s `QAStatusMonitor.report(step=S.ALL)` function. '''
        self.status_mon.report(step)
        if self.status_mon.steps[S.COMPLETE.name]:
            self.completed_ts = datetime.now().replace(microsecond=0).astimezone()
            self.finished()
            self.info()
            print()

    # info
    def info(self):
        ''' Print basic information about this instance of QAMgr. '''
        print('\nQAMgr Info\n' + 10*'=')
        self.patient.info()
        print('Main Directory:\t', self.main_dir+os.sep)
        print('Started:\t', self.started_ts.isoformat())
        if not self.completed_ts:
            print('Completed:\t --')
        else:
            print('Completed:\t', self.completed_ts.isoformat())
        print('')

    # run
    def run(self):
        ''' Attempt to run all QA steps automatically.
              NOTE: Still requires user input to advance certain steps. '''
        print('\n\nRunning QAMgr...\n')
        self.create_subdirs()
        self.create_spreadsheet()
        self.start_2dc()
        self.prepare_report()
        self.info()
        self.status('nodrives')
        # self.finished()

    # finished
    def finished(self):
        print('\nQAMgr Finished!\n')

    # rtplan_summary
    def rtplan_summary(self, verbose: bool = False):
        self.patient.rtplan.summarize(verbose)


# QAStatusMonitor -- monitor and report QAMgr status
class QAStatusMonitor:
    ''' Monitors and reports various status items for an instance of QAMgr.\n
    Attributes
    ----------
    parent :                instance of QAMgr that this object is monitoring
    steps :                 dict of bool
    Methods
    -------
    update() :              update status for desired object(s)
    report() :              print current status report for desired object(s)
    info() :                print summary
    '''

    # ctor
    def __init__(self, parent):
        self.parent = parent
        self.steps = {}
        for step in S:
            if not ( (step == S.ALL) or (step == S.DRIVES) ):
                self.steps[step.name] = False

    # update desired step(s)
    def update(self, step = S.ALL):
        ''' Update the status of the desired step, generally by checking
              whether certain files or folders have been created yet. '''
        mdir = self.parent.main_dir
        match step:
            # drives:
            case S.ALL | S.DRIVES:
                log.debug(' Updating DRIVES...')
                for _ in self.parent.os_info.drive_list:
                    _.update_status()
            # start:
            case S.ALL | S.START:
                log.debug(' Updating START...')
                self.steps[S.START.name] = True
            # create_subdirs:
            case S.ALL | S.CREATE_SUBDIRS:
                log.debug(' Updating CREATE_SUBDIRS...')
                if os.path.isdir( os.path.join(mdir, 'plan') ) and \
                        os.path.isfile( os.path.join(mdir, 'plan', self.parent.patient.rtplan.file) ) and \
                        os.path.isdir( os.path.join(mdir, 'imports') ) and \
                        os.path.isdir( os.path.join(mdir, 'PDFs') ):
                    self.steps[S.CREATE_SUBDIRS.name] = True
                else:
                    self.steps[S.CREATE_SUBDIRS.name] = False
            # create_spreadsheet:
            case S.ALL | S.CREATE_SPREADSHEET:
                log.debug(' Updating CREATE_SPREADSHEET...')
                self.steps[S.CREATE_SPREADSHEET.name] = os.path.isfile(self.parent.final_basename+'.xls')
            # start_2dc:
            case S.ALL | S.START_2DC:
                log.debug(' Updating START_2DC...')
                if os.path.isdir(self.parent.topas_pt_dir) and os.listdir(self.parent.topas_pt_dir):
                    self.steps[S.START_2DC.name] = True
                else:
                    self.steps[S.START_2DC.name] = False
            # prepare_report:
            case S.ALL | S.PREPARE_REPORT:
                log.debug(' Updating PREPARE_REPORT...')
                # self.steps[S.PREPARE_REPORT.name] = os.path.isfile(self.parent.final_basename+'_Combined.pdf')
                combined_pdf_filename = self.parent.final_basename + '_Combined.pdf'
                log.debug(f'  QAStatusMonitor combined_pdf_filename:\t`{combined_pdf_filename}`')
                # self.steps[S.PREPARE_REPORT.name] = os.path.isfile(combined_pdf_filename)
                if os.path.isfile(combined_pdf_filename):
                    self.steps[S.PREPARE_REPORT.name] = True
                    log.debug(f'  combined_pdf_filename exists.')
                else:
                    self.steps[S.PREPARE_REPORT.name] = False
                    log.debug(f'  combined_pdf_filename DNE.')
            # completed:
            case S.ALL | S.COMPLETE:
                log.debug(' Updating COMPLETE...')
                stepd = {}
                for step in S:
                    if not ( (step==S.ALL) | (step==S.DRIVES) or (step==S.COMPLETE) ):
                        stepd[step.name] = self.steps[step.name]
                if all(stepd.values()):
                    self.steps[S.COMPLETE.name] = True
            case _:
                print(f'{step}:\tNOT FOUND')

    # report
    def report(self, step = S.ALL):
        ''' Update ALL status values, then print the status of the requested step(s)
              for single-item steps, also return the corresponding bool value. '''
        # if step == 'nodrives':
        #     for _step in [S.CREATE_SUBDIRS, S.CREATE_SPREADSHEET, S.START_2DC, S.PREPARE_REPORT, S.COMPLETE]:
        #         log.debug(f' Updating {_step.name}...')
        #         self.update(_step)
        # else:
        #     log.debug(f' Updating {step.name}...')
        #     self.update(step)
        steplist = [S.CREATE_SUBDIRS, S.CREATE_SPREADSHEET, S.START_2DC, S.PREPARE_REPORT, S.COMPLETE]
        if not step == 'nodrives':
            steplist.append(S.DRIVES)
        for _step in steplist:
            self.update(_step)
        print('\nQAMgr Status\n' + 12*'~')
        steps = []
        for s in Steps:
            if not s == S.DRIVES:
                steps.append(s)
        statusd = {}
        if not step == S.DRIVES:
            print('Steps:')
        if (step == S.ALL) or (step == 'nodrives'):
            statusd = self.steps
        else:
            for s in steps:
                if step == s:
                    statusd[s.name] = self.steps[step.name]
        print(tab([(' ',k.title(),bool(v)) for k,v in statusd.items()], tablefmt = 'plain'))
        if ( (step == S.ALL) or (step == S.DRIVES) ) and not (step=='nodrives'):
            self.parent.os_info.status()
        print()
        if (not step == S.ALL) and (not step == S.DRIVES) and (not step == 'nodrives'):
            return self.steps[step]

    # info
    def info(self):
        pp(self.__dict__)


# OSInfo -- os, drive, and directory information
class OSInfo:
    ''' Determines and holds information related to operating system,
            drives, and directories.\n
    Attributes
    ----------
    Q :                     MountedDrive object
    L :                     MountedDrive object
    X :                     MountedDrive object
    Z :                     MountedDrive object
    Q_base :                path-like
    L_base :                path-like
    X_base :                path-like
    Z_base :                path-like
    drive_list :            list
    base_path_list :        list
    Methods
    -------
    status() :              print current status
    info() :                alias for status()
    '''

    # ctor
    def __init__(self):
        # mounted drives
        self.Q = MountedDrive('q')
        self.L = MountedDrive('l')
        self.X = MountedDrive('x')
        self.Z = MountedDrive('z')
        self.drive_list = [ self.Q, self.L, self.X, self.Z ]
        [ _.configure() for _ in self.drive_list ]
        # absolute paths
        self.Q_base = os.path.join( self.Q.current_path, *Q_DIR_PARTS )
        self.L_base = os.path.join( self.L.current_path, *L_DIR_PARTS )
        # self.X_base = os.path.join( self.X.current_path, *X_DIR_PARTS )
        self.X_base = self.X.current_path # [None] parts
        self.Z_base = os.path.join( self.Z.current_path, *Z_DIR_PARTS )

    # status
    def status(self):
        ''' Print the status of OSInfo and / or its collection of mounted drives. '''
        print('Drive Status:')
        msg_drive_status = dict()
        for _ in self.drive_list:
            _.update_status()
            if _.status:
                msg_drive_status[_] = 'OK'
            else:
                msg_drive_status[_] = '--'
        print(tab([(' ',k.nickname.upper(),v) for k,v in msg_drive_status.items()], tablefmt = 'plain'))
        print()

    # info -- alias for status
    def info(self):
        self.status()


# MountedDrive -- manage network drives
class MountedDrive:
    ''' Manages mounted network drives.\n
    Attributes
    ----------
    nickname :              str
    nominal_path :          path-like
    current_path :          path-like
    drivehost :             str (hostname) or ipaddress.IPv4Address
    status :                bool
    Methods
    -------
    configure() :           fill empty attributes
    update_status() :       check if accessible
    info() :                print summary
    '''

    # ctor
    def __init__(self, nick = None, nompath = None):
        self.nickname = nick
        self.nominal_path = nompath or nick
        self.current_path = None
        self.drivehost = None
        self.status = False

    # configure
    def configure(self, nompath = None, nick = None):
        ''' Assign various drive parameters. '''
        # basics
        #TODO: generalize/improve nick/path logic
        # self.nominal_path = nompath #TODO
        # if self.nickname and nick:
        #     msg_already_nick = f'Warning: This MountedDrive object is already nicknamed `{self.nickname}`.'
        #     raise UserWarning(msg_already_nick)
        # elif not self.nickname:
        #     msg_no_nick = 'Warning: No nickname assigned to this MountedDrive object.'
        #     raise UserWarning(msg_no_nick)
        # else:
        #     self.nickname = nick
        self.nompath = self.nickname #TODO #TEMP
        # determine actual path #TODO #TEMP: set rather than configure
        if IS_UNIX:
            self.current_path = r'/mnt/' + self.nominal_path.lower()
        else:
            self.current_path = self.nominal_path.upper() + r':'
        #TODO: fill self.drivehost
        self.update_status()

    # update_status
    def update_status(self):
        ''' Check for drive connectivity and update status accordingly. '''
        if CHECK_DRIVES:
            if IS_UNIX:
                self.status = os.path.ismount(self.current_path)
            else:
                self.status = os.path.isdir(self.current_path)
        else:
            self.status = False

    # info
    def info(self):
        pp(self.__dict__)


# Patient -- holds patient's information
class Patient:
    ''' Holds patient info.\n
    Attributes
    ----------
    rtplan :                RTPlan object
    dcinfo :                pydicom.FileDataset
    patient_dir :           path-like
    plan_dir :              path-like
    given :                 str
    family :                str
    mrn :                   str
    plan_name :             str
    final_basename :        str
    summary :               str
    Methods
    -------
    fill_from_dicom() :     parse the input DICOM file and set parameters accordingly
    prepare_subdirs() :     prepare subdirectory structure
    info() :                print current parameters
    '''

    # ctor
    def __init__(self, FILE: str = ''):
        # init
        self.rtplan = None
        self.dcinfo = None
        self.patient_dir = ''
        self.plan_dir = ''
        self.given = ''
        self.family = ''
        self.mrn = ''
        self.final_basename = ''
        self.plan_name = ''
        self.summary = ''
        self.fill_from_dicom(FILE)
        self.prepare_subdirs()

    # fill_from_dicom
    def fill_from_dicom(self, FILE: str = ''):
        ''' Read the input DICOM file and fill parameters based on the dataset. '''
        # find and fill plan
        self.rtplan = RTPlan(FILE)
        self.dcinfo = self.rtplan.dcinfo
        self.prepare_subdirs()
        # patient basics
        self.family = self.dcinfo.PatientName.family_name.title()
        self.given = self.dcinfo.PatientName.given_name.title()
        self.mrn = self.dcinfo.PatientID
        self.plan_name = self.rtplan.plan_name
        if self.family and self.given:
            self.final_basename = self.family.title() + self.given.title()[0] + '_' + self.mrn
        else:
            self.final_basename = 'Patient' + '_' + self.mrn
        self.xls_filename = self.final_basename + '.xls'
        self.fill_summary()

    # prepare_subdirs
    def prepare_subdirs(self):
        ''' Manage patient and plan subdirectories. '''
        self.patient_dir = f'{self.family.upper()}, {self.given.upper()}~{self.mrn}'
        self.plan_dir = f'{self.plan_name}'
        #TODO: check if existing and adjust if needed

    # fill_summary
    def fill_summary(self):
        ''' Fill patient-info summary string. '''
        self.summary += f'\nGiven_Name\t {self.given}\n'
        self.summary += f'Family_Name\t {self.family}\n'
        self.summary += f'MRN\t\t {self.mrn}\n'
        self.summary += f'Plan\t\t {self.plan_name}'
        self.summary += '\n'
        # summary_dict = { 'Given_Name': self.given,
        #         'Family_Name': self.family,
        #         'MRN': self.mrn,
        #         'Plan': self.plan_name }
        # self.summary = tab( [(k,v) for k,v in summary_dict.items()], tablefmt = 'plain' )

    # info (i.e., print summary)
    def info(self):
        ''' Print current patient-info summary. '''
        print('\nPATIENT:'+self.summary)

    # override __repr__
    def __repr__(self):
        return f'{NAME}.Patient object: {self.family}, {self.given}'

    # override __str__
    def __str__(self):
        return f'{NAME}.Patient object: {self.family}, {self.given}'


# RTPlan -- holds patient's plan parameters
class RTPlan:
    ''' Holds patient's plan parameters
    Attributes
    ----------
    file :                  path-like
    realpath :              path-like
    dcinfo :                pydicom.FileDataset
    plan_name :             str
    has_photons :           bool
    has_ions :              bool
    photon_beam_seq :       pydicom.sequence.Sequence of pydicom.dataset.Dataset
    ion_beam_seq :          pydicom.sequence.Sequence of pydicom.dataset.Dataset
    num_photon_fields :     int
    num_ion_fields :        int
    total_fields :          int
    ion_types :             set of str
    photon_lines :          list of str
    ion_lines :             list of str
    Methods
    -------
    # list_beams() :         list photon beams (primarily internal)
    # list_ion_beams() :     list ion beams (primarily internal)
    # summarize() :          print plan summary
    summarize(bool) :        print beam information
    info(bool) :             alias for summarize()
    '''

    # ctor
    def __init__(self, FILE = ''):
        # init
        self.osinfo = None
        self.file = None
        self.dcinfo = None
        self.plan_name = ''
        self.photon_beam_seq = None
        self.ion_beam_seq = None
        self.has_photons = False
        self.has_ions = False
        self.num_photon_fields = 0
        self.num_ion_fields = 0
        self.total_fields = 0
        self.ion_types = None
        # get os info
        self.osinfo = OSInfo()
        # set file
        if FILE:
            self.file = FILE
        else:
            if GUI:
                # gui version:
                from tkinter import filedialog
                if TESTMODE:
                    starting_dir = os.path.join(QA_DIR, 'tps')
                else:
                    starting_dir = self.osinfo.Q_base
                self.file = filedialog.askopenfilename(title = 'QAMGr File Selection', initialdir = starting_dir, filetypes = [('DICOM', '*.dcm')])
            else:
                # cli version:
                rtplan_files = sorted( ls(r'rp.*\.dcm') )
                L = len(rtplan_files)
                match L:
                    case 0:
                        msg_rtplan_notfound = '\nNo RTPLAN files (RP*.dcm) found.'
                        raise FileNotFoundError(msg_rtplan_notfound)
                    case 1:
                        self.file = rtplan_files[0]
                    case _:
                        print('\nAvailable DICOMs:')
                        for k,v in enumerate(rtplan_files):
                            print(f'\t{k} :\t{v}')
                        FILENUM = int( input(f'Please enter a number to choose a file:\t').strip() )
                        while not FILENUM in range(0, len(rtplan_files)):
                            print(file=sys.stderr)
                            msg_invalid_filenum = 'Invalid selection.'
                            log.warn(msg_invalid_filenum)
                            FILENUM = int( input(f'Please enter an integer between 0 and {len(rtplan_files)-1} to chose a file:\t').strip() )
                        self.file = rtplan_files[FILENUM]
        # get file's absolute path
        log.debug(f'RTPlan::file = {self.file}')
        if (self.file == ()) or (not self.file):
            msg_nofile = 'QAManager requires an input RTPLAN DICOM file.'
            raise FileNotFoundError(msg_nofile)
        else:
            self.realpath = os.path.realpath(self.file)
        # read DICOM
        self.dcinfo = pydicom.dcmread(self.file)
        # extract plan name
        self.plan_name = self.dcinfo.RTPlanLabel.upper()
        if not self.plan_name:
            self.plan_name = self.dcinfo.RTPlanName.upper()
        # extract field counts
        self.has_photons = self.dcinfo.__contains__('BeamSequence')
        self.has_ions = self.dcinfo.__contains__('IonBeamSequence')
        sitesetup = re.compile('site ?setup', re.I)
        if self.has_photons:
            self.photon_beam_seq = self.dcinfo.BeamSequence
            if sitesetup.match(self.photon_beam_seq[-1].BeamName):
                self.photon_beam_seq.pop(-1)
            self.num_photon_fields = len(self.photon_beam_seq)
        if self.has_ions:
            self.ion_beam_seq = self.dcinfo.IonBeamSequence
            if sitesetup.match(self.ion_beam_seq[-1].BeamName):
                self.ion_beam_seq.pop(-1)
            self.num_ion_fields = len(self.ion_beam_seq)
        self.total_fields = self.num_photon_fields + self.num_ion_fields
        # extract field info
        if self.has_photons:
            log.debug('YES PHOTONS')
            self.photon_lines = [ ('Beam Name', 'Number', 'Gantry') ]
            for beam in self.photon_beam_seq:
                cp0 = beam.ControlPointSequence[0]
                self.photon_lines.append(
                    ( beam.BeamName, beam.BeamNumber, cp0.GantryAngle )
                )
        elif self.has_ions:
            log.debug('YES IONS')
            ion_type_list = []
            self.ion_lines = [ ('Beam Name', 'Number', 'Gantry') ]
            for beam in self.ion_beam_seq:
                cp0 = beam.IonControlPointSequence[0]
                self.ion_lines.append(
                    ( beam.BeamName, beam.BeamNumber, cp0.GantryAngle )
                )
                ion_type_list.append(beam.RadiationType)
            self.ion_types = set(ion_type_list)
        else:
            log.debug('NO PHOTONS OR IONS')

    # summarize
    def summarize(self, verbose: bool = False):
        '''Print summary of RTPLAN; include details if `verbose` is True.'''
        print('\nRTPlan Summary:\n')
        if verbose:
            if self.has_photons:
                print( tab(self.photon_lines, headers = 'firstrow', tablefmt = 'simple_outline') )
                print()
            elif self.has_ions:
                print( tab(self.ion_lines, headers = 'firstrow', tablefmt = 'simple_outline') )
                print()
            else:
                log.debug('NO PHOTONS OR IONS')
        # print(f'Photon fields:\t\t{self.num_photon_fields}\nIon Fields:\t\t{self.num_ion_fields}\nTotal Fields:\t{self.total_fields}\n') # orig for reference
        sum_lines = [ ('Photon Fields', '', self.num_photon_fields),
                ('Ion Fields', '', self.num_ion_fields),
                ('Total Fields', '', self.total_fields) ]
        if self.has_ions:
            it = self.ion_types.__str__()
            it = it.replace('{','').replace('}','').replace("'","")
            sum_lines.append( ('Ion Types', '', it) )
        summary = tab(sum_lines, tablefmt = 'plain')
        # summary = tab(sum_lines, tablefmt = 'plain', floatfmt = '03f')
        print(summary)
        print()

    # info -- alias for summarize
    def info(self, verbose: bool = False):
        self.summarize(verbose)

## FUNCTIONS

# list_qa_steps
def list_qa_steps():
    strrep = 'QA Steps:'
    for _ in Steps:
        strrep += '\n  ' + _.name
    print(strrep)

# listre -- regex-search a list
def listre( listname: list, restring: str, reflags=0 ) -> list:
    ''' Search a list for items matching a regex. '''
    pat = re.compile(restring, reflags)
    matches = list( filter(pat.match, listname) )
    return matches

# filter-enabled ls()
def ls( regex: str = '', path: str = None ) -> list:
    ''' List files matching a given regex in a given path. '''
    return listre( os.listdir(path), regex, reflags = re.IGNORECASE )

# main
def main():
    ''' Create and run an instance of QAMgr. This function will probably
          be more relevant for testing than for production use. '''
    if GUI:
        m = GuiQAMgr()
    else:
        m = QAMgr()
        m.run()


## DATA

# run as executable
if __name__ == '__main__':
    main()

# mount info
# shared-drive info:
Q_DRIVE = [ 'BJCShared13', 'RADONC-Clinical-DataQ' ]
L_DRIVE = [ 'bjhradonc3', 'physicsqa' ]
X_DRIVE = [ r'va_data$', 'Dicom', 'EXPORTS' ]
Z_DRIVE = [ '10.35.125.188', 'DoseCalc' ]
# relative path info:
Q_DIR_PARTS = [ 'Proton HypersScan Commissioning', 'PatientQA' ]
L_DIR_PARTS = [ 'PatientQA', 'Excel Generator (ALL)' ]
X_DIR_PARTS = [ None ]
Z_DIR_PARTS = [ 'TopasDoseCalc' ]
# test directory
if os.name=='posix': # Linux / Mac
    QA_DIR = '/home/mark/mark_code/qamgr/test'
else:               # Windows
    QA_DIR = r'C:\Users\duvall\Documents\duvall_personal\tests_and_demos\qamgr'

# other constants
LOGGER = log.getLogger()
IS_UNIX = (os.name=='posix')
HOST = socket.gethostname()
# HOST_IP = socket.gethostbyname(HOST)
HOME = os.path.expanduser('~')

# module information
NAME = 'QAMgr'
VERSION_MAJOR = '0'
VERSION_MINOR = '9'
VERSION_BUILD = '0'
VERSION = '.'.join([VERSION_MAJOR, VERSION_MINOR, VERSION_BUILD])
print(f'{NAME} v{VERSION} loaded.\n')
log.debug('GUI = %s\n' % str(GUI))


## all pau!   )

