''' Module for adding a GUI interface (via tkinter) to QAMgr.

Authorship:
duvall@wustl.edu
Siteman Cancer Center,
Washington University School of Medicine
11/2024

'''

##TODO
# - clean up outdated items since migrating to manual tracking

# IDEAS:
# - hacky workaround for live-vars not updating:
#     create an update_gui_labels function
#       and call it, e.g., each time something changes
# THIS WORKED!

#REMINDERS:
# Traceable variable types from tkinter are as follows:
# - BooleanVar
# - IntVar
# - DoubleVar
# - StringVar


## IMPORTS
import math, io, contextlib, pathlib
from qamgr import *
from qamgr.qapdfs import *
import tkinter as tk
from tkinter import font, simpledialog, filedialog, messagebox
from functools import partial, partialmethod
# from ttk...
IS_UNIX = (os.name=='posix')


## CLASSES

# TKVars -- shorthands for traceable tkinter var types
class TKVars():
    ''' Shorthands for traceable tkinter var types\n
    '''
    # values
    B = tk.BooleanVar
    I = tk.IntVar
    D = tk.DoubleVar
    S = tk.StringVar
V = TKVars


# GuiQAMgr
# class GuiQAMgr():
class GuiQAMgr(QAMgr):
    ''' Like QAMgr, but with a GUI.\n
    '''

    def __init__(self, FILE = ''):
        qamgr.GUI = True
        super().__init__(FILE)
        self.steplist = []
        self.root = tk.Tk()
        self.root.bind('<Control-w>', self.close_all)
        self.root.bind('<Escape>', self.close_all)
        # self.root.bind('<Control-w>', self.close_current_window)
        # self.root.bind('<Escape>', self.close_current_window)
        self.root.bind('<Return>', self.handle_return)
        self.defaultFont = font.nametofont('TkDefaultFont')
        self.defaultFont.configure(family = 'arial', size = 14)
        self.root.widgets = []
        self.frames = {}
        self.guisteps = []
        w =  700 # width for the Tk root
        h =  800 # height for the Tk root
        ws = self.root.winfo_screenwidth() # width of the screen
        hs = self.root.winfo_screenheight() # height of the screen
        # x = (ws/2) - (w/2) # single monitor, centered
        x = 3*(ws/4) - (w/2) # single monitor, offset right
        # x = (ws/2) - 3*(w/2) # dual monitors
        y = (hs/2) - (h/2)
        self.root.geometry('%dx%d+%d+%d' % (w, h, x, y))
        self.root.title('QAMgr')

        # header
        l_heading = tk.Label(self.root, text = 'QA Manager', justify = 'center', font = header_font)

        # frames
        f_pt = tk.Frame(self.root)
        f_drives_hdr = tk.Frame(self.root)
        f_drives = tk.Frame(self.root)
        f_steps_hdr = tk.Frame(self.root)
        f_steps = tk.Frame(self.root)
        f_close = tk.Frame(self.root)
        self.frames['patient'] = f_pt
        self.frames['drives_hdr'] = f_drives_hdr
        self.frames['drives'] = f_drives
        self.frames['steps_hdr'] = f_steps_hdr
        self.frames['steps'] = f_steps
        self.frames['close'] = f_close

        # pt and plan info
        l_pt_hdr = tk.Label(f_pt, text = 'Patient Info', font = subheader_font, justify = 'left')
        l_pt = tk.Label(f_pt, text = self.patient.summary, font = standard_font, justify = 'left')
        l_pt_hdr.pack()
        l_pt.pack()
        b_plan_details = tk.Button(f_pt, text = 'Show Plan Details', font = button_font, width = 16, command = self.show_plan_details)
        b_plan_details.pack()

        # drive connectivity
        l_drives = tk.Label(f_drives_hdr, text = 'Network Drives', justify = 'left', font = subheader_font)
        b_refresh_drives = tk.Button(f_drives_hdr, text = 'Refresh', font = button_font, width = 8, command = self.refresh_drives)
        l_drives.pack(padx = 16, side = tk.LEFT)
        b_refresh_drives.pack(padx = 8, side = tk.RIGHT)
        self.drive_labels = {}
        drivenum = 0
        for _ in self.os_info.drive_list:
            l_row = math.floor(drivenum / 2)
            l_col = 2 * (drivenum % 2)
            log.debug(f'  Drive grid:  ({l_row}, {l_col})')
            l_drive_name = tk.Label(f_drives, text = _.nickname.upper(), font = standard_font, justify = 'left', width = 4)
            l_drive_value = tk.Label(f_drives, fg = 'white', bg = 'red', font = standard_font, justify = 'left', width = 4, text = '--')
            l_drive_name.grid(row = l_row, column = l_col, padx = 8, pady = 4)
            l_drive_value.grid(row = l_row, column = l_col+1, padx = 8, pady = 4)
            self.drive_labels[_.nickname] = l_drive_value
            drivenum += 1

        # steps
        l_steps_hdr = tk.Label(f_steps_hdr, text = 'Steps', justify = 'left', font = subheader_font)
        b_refresh = tk.Button(f_steps_hdr, text = 'Refresh', font = button_font, width = 8, command = self.refresh)
        l_steps_hdr.pack(padx = 16, side = tk.LEFT)
        b_refresh.pack(padx = 8, side = tk.RIGHT)
        stepnum = 0
        for _ in S:
            if (_ == S.ALL) or (_ == S.DRIVES):
                pass
            else:
                gui_step = GuiStep(self, _) 
                self.steplist.append(gui_step)
                gui_step.grid(row = stepnum)
                self.guisteps.append(gui_step)
                stepnum += 1

        # close button
        b_close = tk.Button(f_close, text = 'Close', font = button_font, width = 8, command = self.root.destroy)
        b_close.pack()
        self.b_close = b_close

        # pack widgets
        l_heading.pack(side = tk.TOP, pady = 8)
        f_pt.pack(pady = 4)
        f_drives_hdr.pack(pady = 8)
        f_drives.pack(pady = 4)
        f_steps_hdr.pack(pady = 8)
        f_steps.pack(pady = 4)
        f_close.pack(side = tk.BOTTOM, pady = 8)

        # Start the GUI event loop
        self.refresh_drives()
        self.refresh()
        self.root.update()
        self.root.update_idletasks()
        # self.root.focus()
        self.root.mainloop()

    # refresh
    def refresh(self):
        log.info('  Refreshing step completion status.\n')
        for _ in self.guisteps:
            if self.status_mon.steps[_.step.name]:
                _.l_value.config(bg = 'green', text = 'Complete')
            else:
                _.l_value.config(bg = 'red', text = 'Pending')
        if self.status_mon.steps[S.COMPLETE.name]:
            self.finished_gui()
        self.root.update()
        self.root.update_idletasks()

    # refresh_drives
    def refresh_drives(self):
        log.info('  Refreshing drive connectivity status.\n')
        for _ in self.os_info.drive_list:
            _.update_status()
            l_drive_value = self.drive_labels[_.nickname]
            if _.status:
                l_drive_value.config(bg = 'green', text = 'OK')
            else:
                l_drive_value.config(bg = 'red', text = '--')

    # prepare final-report PDF
    def prepare_report(self):
        ''' Combine patient\'s spreadsheet and `PDFs` files into a single,
              compressed, final-report PDF ready for uploading into Aria. '''
        print('Preparing final report...')
        # QAPDFS VERSION:
        origwd = os.getcwd()
        log.debug(f'  origwd:\t{origwd}\n  self.main_dir:\t{self.main_dir}')
        log.debug(f'  self.final_basename = `{self.final_basename}`')
        if os.path.isdir(self.main_dir):
            os.chdir(self.main_dir)
            log.debug(f'  Entering directory `{self.main_dir}`')
        else:
            msg_notadir = f'Directory {self.main_dir} does not exist; please run create_subdirs first.'
            raise NotADirectoryError(msg_notadir)
        if TESTMODE:
            xls_pdf_filename = self.final_basename + '.pdf'
            pdf_filename = self.final_basename + '_Combined.pdf'
            srcdir = os.path.dirname(self.patient.rtplan.realpath)
            # testspr = os.path.join(srcdir, 'spreadsheet.pdf') 
            # testspr = os.path.join(srcdir, self.final_basename + '.pdf') 
            testspr_dir = os.path.realpath( os.path.join(srcdir, '..', 'tps') )
            testspr = os.path.join( testspr_dir, os.path.basename(xls_pdf_filename) )
            if os.path.isfile(testspr):
                shutil.copy( testspr, xls_pdf_filename )
            else:
                log.error(f'  File testspr = `{testspr}` not found.')
        if os.name == 'posix':
            r = Report()
            r.run()
        else:
            pathlib.Path.touch(pdf_filename)
        self.status_mon.steps[S.PREPARE_REPORT.name] = True
        os.chdir(origwd)
        log.debug(f'  Entering directory `{origwd}`')
        if not __name__ == '__main__':
            self.status()

    # close_all
    def close_all(self, _):
        self.root.destroy()

    # # close_current_window
    # def close_current_window(self, _):
    #     current_widget = self.root.focus_get()
    #     # while not ( isinstance(current_widget, tk.Toplevel) or isinstance(current_widget, tk.Tk) ):
    #     # while not issubclass(type(current_widget), tk.Misc):
    #     while current_widget.master and (not isinstance(type(current_widget), tk.Misc)) and (not current_widget == self.root):
    #         log.debug(f'  current_widget:\t{current_widget}')
    #         current_widget = current_widget.master
    #     log.debug(f'  Final current_widget:\t{current_widget}')
    #     current_widget.destroy()

    # handle_return
    def handle_return(self, _):
        focused = self.root.focus_get()
        if isinstance(focused, tk.Button):
            focused.invoke()

    # finished_gui
    def finished_gui(self):
        response_finished = messagebox.showinfo(title = 'QAMgr Finished!', message = 'QA Complete!')
        self.root.update()
        self.root.update_idletasks()
        log.debug(f'  response_finished = {response_finished}')
        if response_finished == messagebox.OK:
            self.root.destroy()

    # show_plan_details
    def show_plan_details(self):
        p = self.patient.rtplan
        captured_summary = io.StringIO()
        with contextlib.redirect_stdout(captured_summary):
            p.summarize(True)
        w_pd = tk.Toplevel(master = self.root)
        l_pd_hdr = tk.Label(w_pd, text = 'Plan Details', font = subheader_font).pack(pady = 8)
        l_pd = tk.Label(w_pd, text = captured_summary.getvalue(), font = mono_font, justify = 'left').pack(padx = 16, pady = 8)
        b_pd_close = tk.Button(w_pd, text = 'Close', font = button_font, width = 8, command = w_pd.destroy).pack(pady = 8)
        w_pd.focus()
        # w_pd.bind('<Control-w>', w_pd.destroy)
        # w_pd.bind('<Escape>', w_pd.destroy)
        # w_pd.bind('<Return>', self.root.handle_return)


# GuiStep
# Note -- Each step should have:
# - A parent GuiQAMgr object
# - A "master" (i.e., root) tk.Tk object/window
# - A step ID (e.g., S.CREATE_SUBDIRS)
# - A "Name" label = step.name.title()
# - A "True/False" tk.Label to show its status
# - An "Execute" button to call the respective method from its parent
class GuiStep:

    # ctor
    def __init__(self, parent, step):   # parent = GuiQAMgr, step = Steps.?
        self.parent = parent
        self.root = parent.root
        self.frame = self.parent.frames['steps']
        self.step = step
        self.l_name = None
        self.l_value = None
        self.cmd = None
        self.execute_button = None
        self.l_name = tk.Label(self.parent.frames['steps'], text = step.name.title(), font = standard_font, justify = 'left')
        self.l_value = tk.Label(self.parent.frames['steps'], width = 16, font = standard_font, fg = 'white', bg = 'red', text = 'False')
        match step:
            case Steps.CREATE_SUBDIRS:
                self.cmd = self.parent.create_subdirs
            case Steps.CREATE_SPREADSHEET:
                self.cmd = self.parent.create_spreadsheet
            case Steps.START_2DC:
                self.cmd = self.parent.start_2dc
            case Steps.PREPARE_REPORT:
                self.cmd = self.parent.prepare_report
        self.execute_button = tk.Button( self.parent.frames['steps'], text = 'Execute', font = button_font, command = lambda: [self.cmd(), self.parent.refresh()] )
        # self.execute_button.bind('<Return>', self.execute_button.invoke())

    # pack
    def pack(self, *args, **kwargs):
        self.l_name.pack(*args, **kwargs, side = tk.LEFT, padx = 16)
        self.l_value.label.pack(*args, **kwargs, side = tk.LEFT, padx = 16)
        if not self.step == S.START:
            self.execute_button.pack(*args, **kwargs, side = tk.LEFT, padx = 16)

    # grid
    def grid(self, *args, **kwargs):
        self.l_name.grid(*args, **kwargs, column = 0, padx = 8)
        self.l_value.grid(*args, **kwargs, column = 1, padx = 8)
        if not ( self.step == S.START or self.step == S.COMPLETE):
            self.execute_button.grid(*args, **kwargs, column = 2, padx = 8)


## FUNCTIONS

# callbacks:
# update_bool_label
def update_bool_label(var, label, _1 = None, _2 = None, _3 = None): # var = tk.BooleanVar, label = tk.Label
    if var.get():
        label.config(text = 'YES')
        label.config(bg = 'green', fg = 'white')
    else:
        label.config(text = ' NO')
        label.config(bg = 'red', fg = 'white')


## DATA

if IS_UNIX:
    serif_font_family = 'DejaVu Serif'
    sans_font_family = 'DejaVu Sans'
    mono_font_family = 'DejaVu Sans Mono'
    font_sizes = (20, 16, 14, 14)
else:
    serif_font_family = 'Georgia'
    sans_font_family = 'Calibri'
    mono_font_family = 'Cascadia Code'
    font_sizes = (18, 14, 12, 12)
header_font = (serif_font_family, font_sizes[0], font.BOLD)
subheader_font = (serif_font_family, font_sizes[1], font.BOLD)
button_font = (sans_font_family, font_sizes[2])
standard_font = (sans_font_family, font_sizes[2])
mono_font = (mono_font_family, font_sizes[3])



## all pau!   )
